%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This function is used to define dynamics of individual components
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ***** angular velocity of basis *****
omega0 = 2*pi*60; %[rad/s]

% ***** settings of generators *****
gpar = cell(1,NG);
index= 0;

for j=1:NG
  %set of indices of state variables of the generator+PSS+turbine
  gpar{j}.si  = [index+1:index+(4+3)]; 
  index = index + (4+3);    
  
  %dimension
  gpar{j}.n = length(gpar{j}.si); 

  % ***************** sync. generators ******************
  %define: parameters of sync. generators
  xl = mac_con(j,4);%xl
  ra = mac_con(j,5);%ra
  xd = mac_con(j,6);%xd
  xq = mac_con(j,11);%xq
  xdp = mac_con(j,7);%xd'
  xqp = mac_con(j,12);%xq'
  xdpp = mac_con(j,8);%xd''
  xqpp = mac_con(j,13);%xq''
  Tdp = mac_con(j,9); %Tdo'
  Tqp = mac_con(j,14); %Tqo'
  Tdpp= mac_con(j,10); %Tdo'' 
  Tqpp= mac_con(j,15);%Tqo'' 
  M  = mac_con(j,16);%M
  D  = mac_con(j,17);%D  
  Ta = exc_con(:,5);%Ta
  Ka = exc_con(:,4)/200*20; % exciter gain
  
  %save several parameters
  gpar{j}.ra = ra;
  gpar{j}.xdp = xdp;
  gpar{j}.xq = xq;
  gpar{j}.xd = xd;
  
  % ***************** pss ******************
  Kp =pss_con(j,3);
  Tw =pss_con(j,4);
  Tn1=pss_con(j,5);
  Td1=pss_con(j,6);
  Tn2=pss_con(j,7);
  Td2=pss_con(j,8);
  
  %ABCD matrices of PSS
  pss{j}.Apsi = [-1/Tw, 0, 0; ...
    -Kp/(Td1*Tw)*(1-Tn1/Td1), -1/Td1, 0;...
    -(Kp*Tn1)/(Td1*Td2*Tw)*(1-Tn2/Td2), 1/Td2*(1-Tn2/Td2), -1/Td2];

  pss{j}.Bpsi = [1/Tw; ...
    Kp/(Td1*Tw)*(1-Tn1/Td1);...
    (Kp*Tn1)/(Td1*Td2*Tw)*(1-Tn2/Td2)];  

  pss{j}.Cpsi = [-(Kp*Tn1*Tn2)/(Td1*Td2*Tw), Tn2/Td2, 1];

  pss{j}.Dpsi = (Kp*Tn1*Tn2)/(Td1*Td2*Tw);

  % ==== differential equations ====
  %xk := [delta; omega; e; ef; xPSS; eta]
  %external input = (pm, vabs, vang, vr)
  dae{j}.dxk = @(xk, pm, vabs, vang, vr, vfdstar,u) [...
	omega0*xk(2); ...
    (1/M) * (pm - D*xk(2) - vabs*xk(3)/xdp*sin(xk(1)-vang) + vabs^2/2*(1/xdp-1/xq)*sin(2*(xk(1)-vang))); ...
    (1/Tdp) * (xk(4)-xd/xdp*xk(3)+(xd-xdp)/xdp*vabs*cos(xk(1)-vang)); ...
    (1/Ta) * (-(xk(4)-vfdstar) - Ka*(vabs - vr - (u + pss{j}.Cpsi*xk(5:7) + pss{j}.Dpsi*xk(2)))); ...
    pss{j}.Apsi*xk(5:7) + pss{j}.Bpsi*xk(2);
    ];    
  
  % ==== output equations ====
  dae{j}.pow = @(xk, vabs, vang)  [
    xk(3)*vabs/xdp*sin(xk(1)-vang) - vabs^2/2*(1/xdp-1/xq)*sin(2*(xk(1)-vang)); ...
    xk(3)*vabs/xdp*cos(xk(1)-vang) - vabs^2*((sin(xk(1)-vang))^2/xq+(cos(xk(1)-vang))^2/xdp); ...
    ];
  
  dae{j}.cur = @(xk, V, th)  [
    (xk(3)*sin(xk(1))-V*sin(th))/xdp - (1/xdp - 1/xq)*V*sin(xk(1)-th)*cos(xk(1)); ...
    -(xk(3)*cos(xk(1))-V*cos(th))/xdp - (1/xdp - 1/xq)*V*sin(xk(1)-th)*sin(xk(1))];

  dae{j}.pssU = @(xk,u) u + pss{j}.Cpsi*xk(5:7) + pss{j}.Dpsi*xk(2); 
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Define PV
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
for j=1:NP
  pv{j}.n = 7; %consider AC/DC
  pv{j}.m = 2; %retrofit control input dimension
  
  %Pbase = pv_con(j,1)*1e6; 
  Pbase = mpc.baseMVA*1e6; 
  Ppvbase = pv_con(j,1)*1e6; 
  Zbase = pv_con(j,2)^2/Ppvbase; 
  Ibase  = (Ppvbase/pv_con(j,2));
  Lg = pv_con(j,3)/Zbase*omega0*(Pbase/Ppvbase); 
  Rg = pv_con(j,4)/Zbase*(Pbase/Ppvbase); 
  Gdc = pv_con(j,5)*Zbase*(Pbase/Ppvbase); 
  Cdc = pv_con(j,6)*1e-6*Zbase*omega0*(Pbase/Ppvbase); 
  Rpv = pv_con(j,8)/Zbase*(Pbase/Ppvbase); 
  Vpv = pv_con(j,9)/pv_con(j,2);
  
  Tg = pvp(j,1); 
  Kp_dg = pvp(j,2); 
  Ki_dg = pvp(j,3); 
  Kp_qg = pvp(j,4); 
  Ki_qg = pvp(j,5); 
  m_max = pvp(j,6);   
  
  pv{j}.Lg = Lg;
  pv{j}.Rg = Rg;
  pv{j}.Gdc = Gdc;
  pv{j}.Cdc = Cdc;
  pv{j}.Rpv = Rpv;
  pv{j}.Vpv_s = Vpv;
  pv{j}.Vdcp_s = pv_con(j,10)/pv_con(j,2);
  pv{j}.ipv_s = (pv{1}.Vpv_s - pv{1}.Vdcp_s)/pv{1}.Rpv; 
  pv{j}.P_s =pv{j}.ipv_s*pv{j}.Vdcp_s; 
  
  % ====== AC/DC =======
  % x := [idg; iqg; chi_d; chi_q; Zeta_dg; Zeta_qg] (i, inner, outer)
  pv{j}.con = @(x, vdc, vgrid, P_s, Q_s, mG) [
    (omega0/Lg)*([-Rg, Lg; -Lg, -Rg]*x(1:2) + vgrid - mG/2*vdc); 
    (1/Tg)*(Kp_dg*(P_s + gamma_pv(j)*vgrid'*[x(1);x(2)]) + x(5) - x(1)); 
    (1/Tg)*(Kp_qg*(Q_s + gamma_pv(j)*vgrid'*[-x(2); x(1)]) + x(6) - x(2)); 
    Ki_dg*(P_s + gamma_pv(j)*vgrid'*[x(1);x(2)]); 
    Ki_qg*(Q_s + gamma_pv(j)*vgrid'*[-x(2); x(1)]); 
    ];
  
  pv{j}.mG = @(x, vdc, vgrid, P_s, Q_s, u) [
    max(min(2/vdc*(u(1) + vgrid(1) + Lg*x(2) - Rg*x(3) - Lg/(omega0*Tg)*(Kp_dg*(P_s + gamma_pv(j)*vgrid'*[x(1);x(2)]) + x(5) - x(1))),m_max),-m_max);
    max(min(2/vdc*(u(2) + vgrid(2) - Lg*x(1) - Rg*x(4) - Lg/(omega0*Tg)*(Kp_qg*(Q_s + gamma_pv(j)*vgrid'*[-x(2); x(1)]) + x(6) - x(2))),m_max),-m_max);
    ];
  
  % ====== DC-Link =======
  pv{j}.DC = @(ig, idc, vgrid, vdc) [
    omega0/Cdc*((ig'*vgrid + idc*vdc - Rg*(ig'*ig))/(2*vdc) - Gdc*vdc);
    ];
  
  % **** dynamics ****
  % x := [idg; iqg; chi_d; chi_q; Zeta_dg; Zeta_qg; vdc] (i, inner, outer, dc)
  % u = [u for AC/DC]
  pv{j}.f = @(x, vgrid, P_s, Q_s, Spv, u) [
    pv{j}.con(x(1:6), x(7), vgrid, P_s, Q_s, pv{j}.mG(x(1:6), x(7), vgrid, P_s, Q_s, u));
    pv{j}.DC(x(1:2), (Spv*Vpv - Spv^2*x(7))/Rpv, vgrid, x(7));  
    ];
  pv{j}.pow = @(vgrid, x) -gamma_pv(j)*([vgrid(1), vgrid(2); vgrid(2), -vgrid(1)]*x(1:2));
  pv{j}.cur = @(x) -gamma_pv(j)*x(1:2); 

end

% total dimension of solar farms
if NP>0
  net.npv = sum(t_cell(pv,'n'));
else
  net.npv = 0;
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Define wind power plant 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
for j=1:NW

  win{j}.n = 18; %consider B2B 
  win{j}.m = 4; %retrofit control input dimension

  %if Outer RSC controller is P
  if RSC_flag == 0
    win{j}.n = win{j}.n - 2;
  end
  
  % if battery is integrated
  if sto_flag == 1
    win{j}.n = win{j}.n + 2;
    win{j}.m = win{j}.m + 1;
  end
  
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % define parameters 
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %Pbase = wind_con(j,19)*1e6; 
  Pbase = mpc.baseMVA*1e6;   

  Tbase = Pbase/(omega0/2); 
  Jr = wind_con(j,6)*(omega0/2)/Tbase;
  Br = wind_con(j,7)*(omega0/2)/Tbase;
  Jg = wind_con(j,8)*(omega0/2)/Tbase;
  Bg = wind_con(j,9)*(omega0/2)/Tbase;
  Bdt = wind_con(j,10)*(omega0/2)/Tbase;
  Kd = wind_con(j,11)/Tbase;
  Ng = wind_con(j,12);
  
  Cp = 0.410955856214939; 
  win{j}.pa = 0.5*Cp*wind_con(j,3)*wind_con(j,4)/Pbase; %(wind speed)^3 is multiplied later
  win{j}.Ng = Ng;
  win{j}.Bl = Br;
  win{j}.Br = Bg;
  win{j}.Kc = Kd;
  
  %ABC matrices of turbine
  win{j}.Aeta = [
    -(Bdt + Br)/Jr,  Bdt/(Jr*Ng), -Kd/Jr; ...
    Bdt/(Jg*Ng), -(1/Jg)*(Bdt/(Ng^2) + Bg), Kd/(Jg*Ng); ...
    omega0/2, -(omega0/2)/Ng, 0];
  
  win{j}.Beta = [1/Jr;0;0]; % *pa
  win{j}.Reta = [0;-1/Jg;0];% *tgc

  %Dynamics of turbine
  win{j}.mech = @(xmech, tg, windspeed) win{j}.Aeta*xmech + win{j}.Beta*(windspeed^3/xmech(1))*win{j}.pa + win{j}.Reta*tg;

  % ***************** Electrical Subsystems ******************
  Pwbase = wind_con(j,19)*1e6; 
  p = wind_con(j,13); 
  Xs = wind_con(j,14)*(Pbase/Pwbase); % for grid base 
  Xr = wind_con(j,15)*(Pbase/Pwbase); 
  Xm = wind_con(j,16)*(Pbase/Pwbase); 
  Rs = wind_con(j,17)*(Pbase/Pwbase); 
  Rr = wind_con(j,18)*(Pbase/Pwbase); 
  Shm = 1; 
  K1 = Xs*Xr-Xm^2;
    
  %save several parameters
  win{j}.p = p;
  win{j}.Xm = Xm;
  win{j}.Xs = Xs;
  win{j}.Xr = Xr;
  win{j}.Rs = Rs;
  win{j}.Rr = Rr;
  win{j}.Shm = Shm;
    
  % ABC matrices
  win{j}.Axi = @(omega_r) (1/K1)*[
    -Rr*Xs, K1-omega_r*Xs*Xr, Rs*Xm, -omega_r*Xs*Xm;...
    -K1+omega_r*Xs*Xr, -Rr*Xs, omega_r*Xs*Xm, Rs*Xm;...
    Rr*Xm, omega_r*Xr*Xm, -Rs*Xr, K1+omega_r*Xm^2;...
    -omega_r*Xr*Xm, Rr*Xm, -K1-omega_r*Xm^2, -Rs*Xr];
  
  % for vs := [vds, vqs]^T
  win{j}.Bxi = (1/K1) * [
    Xm, 0;...
    0, Xm;...
    -Xr, 0;...    
    0, -Xr];
  
  % for [vdr, vqr]^T, control input
  win{j}.Rxi = (1/K1) * [
    -Xs, 0;...
    0, -Xs;...
    Xm,0;...    
    0, Xm];

  % DFIG dynamics 
  % vgrid := Shm*[vds;vqs],  vr := [vdr;vqr]
  win{j}.DFIG = @(wr,cur,vgrid,vr) win{j}.Axi(wr)*cur + win{j}.Rxi*vr+ win{j}.Bxi*Shm*vgrid; 
  
  % torque
  win{j}.tg = @(xi) Xm*(xi(3)*xi(2)-xi(4)*xi(1));
  
  % transformation from i to  psi = [pdr; pqr; pds; pqs]
  % Apsi = inv(T)* A * T;
  T =   EYE(4,[1,3,2,4])*[
    -Xs/(Xr*Xs - Xm^2),  Xm/(Xr*Xs - Xm^2),                 0,                  0;
    Xm/(Xr*Xs - Xm^2), -Xr/(Xr*Xs - Xm^2),                  0,                  0;
    0,                  0, -Xs/(Xr*Xs - Xm^2),  Xm/(Xr*Xs - Xm^2);
    0,                  0,  Xm/(Xr*Xs - Xm^2), -Xr/(Xr*Xs - Xm^2);
    ]*EYE(4,[1,3,2,4])';

  win{j}.T = eye(win{j}.n); 
  win{j}.T(4:7, 4:7) = T; 
  
  % ***************** B2B Converters ******************  
  Zbase = wind_con(j,20)^2/(wind_con(j,19)*1e6); 
  Lg = wind_con(j,22)/Zbase*omega0*(Pbase/Pwbase); 
  Rg = wind_con(j,23)/Zbase*(Pbase/Pwbase); 
  Gdc = wind_con(j,24)*Zbase*(Pbase/Pwbase); 
  Cdc = wind_con(j,25)*1e-6*Zbase*omega0*(Pbase/Pwbase); 
  Shl = wind_con(j,21)/wind_con(j,20);   
  win{j}.vdc_s = winp(j,10)/wind_con(j,20); 
  
  Tg = winp(j,1); 
  Kp_dg = winp(j,2); 
  Ki_dg = winp(j,3); 
  Kp_qg = winp(j,4); 
  Ki_qg = winp(j,5); 
  Kp_dr = winp(j,6); 
  Ki_dr = winp(j,7); 
  Kp_qr = winp(j,8); 
  Ki_qr = winp(j,9);   
  m_max = winp(j,13);   

  win{j}.Lg = Lg;
  win{j}.Rg = Rg;
  win{j}.Gdc = Gdc;
  win{j}.Cdc = Cdc;
  win{j}.Shl = Shl;

  % ====== GSC =======
  % x := [idg; iqg; Didg; Diqg; Zeta_dg; Zeta_qg]
  win{j}.GSC = @(x, vgrid, vdc, vdc_s, Q_s, mG) [
    (omega0/Lg)*([-Rg, Lg; -Lg, -Rg]*x(1:2) + Shl*vgrid - mG/2*vdc); 
    (1/Tg)*(Kp_dg*(vdc_s - vdc) + x(5) - x(1)); 
    (1/Tg)*(Kp_qg*(Q_s - gamma(j)*Shl*vgrid'*[-x(2); x(1)]) + x(6) - x(2)); 
    Ki_dg*(vdc_s - vdc); 
    Ki_qg*(Q_s - gamma(j)*Shl*vgrid'*[-x(2); x(1)]); 
    ];

  win{j}.mG = @(x, vgrid, vdc, vdc_s, Q_s, u) [
    max(min(2/vdc*(u(1) + Shl*vgrid(1) + Lg*x(2) - Rg*x(3) - Lg/(omega0*Tg)*(Kp_dg*(vdc_s - vdc) + x(5) - x(1))),m_max),-m_max);
    max(min(2/vdc*(u(2) + Shl*vgrid(2) - Lg*x(1) - Rg*x(4) - Lg/(omega0*Tg)*(Kp_qg*(Q_s - gamma(j)*Shl*vgrid'*[-x(2); x(1)]) + x(6) - x(2))),m_max),-m_max);
    ];
  % ====== RSC =======
  win{j}.RSC = @(ir, ir_ref) [
    Ki_dr*(ir(1) - ir_ref(1));
    Ki_qr*(ir(2) - ir_ref(2));    
    ]; 

  win{j}.ir_ref = @(OC, wr, vgrid, vgrid_s, wr_s, iqr_s) [
    winp(j,14)*Shm*(norm(vgrid) - norm(vgrid_s)) + OC(1);
    winp(j,16)*(wr - wr_s) + OC(2);
    ];
  win{j}.OC = @(wr, vgrid, vgrid_s, wr_s) [
    winp(j,15)*Shm*(norm(vgrid) - norm(vgrid_s));
    winp(j,17)*(wr - wr_s);
    ];
  % x:= [Zeta_dr; Zeta_qr]
  win{j}.mR = @(x, ir, ir_ref, vdc, u) [
    max(min(2/vdc*(Kp_dr*(ir(1) - ir_ref(1)) + x(1) + u(1)),m_max),-m_max);
    max(min(2/vdc*(Kp_qr*(ir(2) - ir_ref(2)) + x(2) + u(2)),m_max),-m_max);
    ];

  % ====== DC-Link =======
  win{j}.DC = @(ig, ir, vr, vgrid, vdc, idc) [
    omega0/Cdc*((ig'*Shl*vgrid + ir'*vr - Rg*(ig'*ig))/(2*vdc) - Gdc*vdc + 1/2*idc);
    ];
  

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % energy storage system
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 
  Lb = sto_con(j,2)/Zbase*omega0*(Pbase/Pwbase); 
  Rb = sto_con(j,4)/Zbase*(Pbase/Pwbase); 
  Gb = sto_con(j,3)*Zbase*(Pbase/Pwbase); 
  Cb = sto_con(j,1)*1e-6*Zbase*omega0*(Pbase/Pwbase); 
  Ss = sto_con(j,5); 
  
  win{j}.Lb = Lb;
  win{j}.Rb = Rb;
  win{j}.Gb = Gb;
  win{j}.Cb = Cb;
  win{j}.Ss = Ss;
  
  % x := [vb, idcP]
  win{j}.bat = @(x,vdc,uS)[
    omega0/Cb*(-x(2) - Gb*x(1));
    omega0/Lb*(x(1) - Rb*x(2) - max((Ss + uS), sto_con(j,6))*vdc);
    ];
  win{j}.idc = @(idcP,uS) max((Ss + uS), sto_con(j,6))*idcP; 
  
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % equations
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 
  % ***************** differential equation *****************
  if RSC_flag == 1
    % xwin := [eta^T, xi^T, xB2B^T]^T 
    %    where xi = [idr iqr ids iqs],
    %         xB2B = [idg; iqg; Didg; Diqg; Zeta_dg; Zeta_qg; Zeta_dr;Zeta_qr; OC_d; OC_q; vdc]
    %               u = [u for mdG, mqG, mdR, mqR]
    win{j}.f = @(x, vgrid, wr_s, ir_s, vdc_s, Q_s, vgrid_s, windspeed, u) [
      win{j}.mech(x(1:3), win{j}.tg(x(4:7)), windspeed);
      win{j}.DFIG(x(2),x(4:7),vgrid, win{j}.mR(x(14:15), x(4:5), win{j}.ir_ref(x(16:17), x(2), vgrid, vgrid_s, wr_s, ir_s(2)), x(18), [u(3);u(4)])/2*x(18));
      win{j}.GSC(x(8:13), vgrid, x(18), vdc_s, Q_s, win{j}.mG(x(8:13), vgrid, x(18), vdc_s, Q_s, [u(1);u(2)]));
      win{j}.RSC(x(4:5), win{j}.ir_ref(x(16:17), x(2), vgrid, vgrid_s, wr_s, ir_s(2)));
      win{j}.OC(x(2), vgrid, vgrid_s, wr_s); 
      win{j}.DC(x(8:9), x(4:5), win{j}.mR(x(14:15), x(4:5), win{j}.ir_ref(x(16:17), x(2), vgrid, vgrid_s, wr_s, ir_s(2)), x(18), [u(3);u(4)])/2*x(18), vgrid, x(18), 0);
      ];    
  else
    % xwin := [eta^T, xi^T, xB2B^T]^T 
    %    where xi = [idr iqr ids iqs],
    %         xB2B = [idg; iqg; Didg; Diqg; Zeta_dg; Zeta_qg; Zeta_dr;Zeta_qr; vdc]
    %               u = [u for mdG, mqG, mdR, mqR]
    win{j}.f = @(x, vgrid, wr_s, ir_s, vdc_s, Q_s, vgrid_s, windspeed, u) [
      win{j}.mech(x(1:3), win{j}.tg(x(4:7)), windspeed);
      win{j}.DFIG(x(2),x(4:7),vgrid, win{j}.mR(x(14:15), x(4:5), win{j}.ir_ref(ir_s, x(2), vgrid, vgrid_s, wr_s, ir_s(2)), x(16), [u(3);u(4)])/2*x(16));
      win{j}.GSC(x(8:13), vgrid, x(16), vdc_s, Q_s, win{j}.mG(x(8:13), vgrid, x(16), vdc_s, Q_s, [u(1);u(2)]));
      win{j}.RSC(x(4:5), win{j}.ir_ref(ir_s, x(2), vgrid, vgrid_s, wr_s, ir_s(2)));
      win{j}.DC(x(8:9), x(4:5), win{j}.mR(x(14:15), x(4:5), win{j}.ir_ref(ir_s, x(2), vgrid, vgrid_s, wr_s, ir_s(2)), x(16), [u(3);u(4)])/2*x(16), vgrid, x(16), 0);
      ];        
    if sto_flag == 1
      % xwin := [eta^T, xi^T, xB2B^T]^T 
      %    where xi = [idr iqr ids iqs],
      %         xB2B = [idg; iqg; Didg; Diqg; Zeta_dg; Zeta_qg; Zeta_dr;Zeta_qr; vdc; vb; idcP]
      %               u = [u for mdG, mqG, mdR, mqR, S]
      win{j}.f = @(x, vgrid, wr_s, ir_s, vdc_s, Q_s, vgrid_s, windspeed, u) [
	win{j}.mech(x(1:3), win{j}.tg(x(4:7)), windspeed);
	win{j}.DFIG(x(2),x(4:7),vgrid, win{j}.mR(x(14:15), x(4:5), win{j}.ir_ref(ir_s, x(2), vgrid, vgrid_s, wr_s, ir_s(2)), x(16), [u(3);u(4)])/2*x(16));
	win{j}.GSC(x(8:13), vgrid, x(16), vdc_s, Q_s, win{j}.mG(x(8:13), vgrid, x(16), vdc_s, Q_s, [u(1);u(2)]));
	win{j}.RSC(x(4:5), win{j}.ir_ref(ir_s, x(2), vgrid, vgrid_s, wr_s, ir_s(2)));
	win{j}.DC(x(8:9), x(4:5), win{j}.mR(x(14:15), x(4:5), win{j}.ir_ref(ir_s, x(2), vgrid, vgrid_s, wr_s, ir_s(2)), x(16), [u(3);u(4)])/2*x(16), vgrid, x(16), win{j}.idc(x(18),u(5)));
	win{j}.bat(x(17:18), x(16), u(5)); 
	];        	
    end
  end
    
  % **** output ****
  win{j}.pow = @(vgrid, x) gamma(j)*(Shm*[vgrid(1), vgrid(2); vgrid(2), -vgrid(1)]*x(6:7)-Shl*[vgrid(1), vgrid(2); vgrid(2), -vgrid(1)]*x(8:9));
  win{j}.cur = @(x) gamma(j)*(Shm*x(6:7) - Shl*x(8:9)); 
  win{j}.powG = @(vgrid, x) gamma(j)*[Shl*vgrid'*[x(8); x(9)]; Shl*vgrid'*[-x(9); x(8)]];   
end
  

% total dimension of wind farms
if NW>0
  net.nw = sum(t_cell(win,'n'));
else
  net.nw = 0;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% settings of network
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
net.ng = sum(t_cell(gpar,'n')); %dimension of state of syncgen
net.n = net.ng + net.nw + net.npv; %dimension of state of syncgen+wind+storage

%indices of voltage of 
net.EGi= [net.n+1: net.n+2*NG]; %generators
net.ELi= [net.n+2*NG+1 : net.n+2*NG+2*NL]; %loads
net.EWi= [net.n+2*NG+2*NL+1 : net.n+2*NG+2*NL+2*NW]; %wind plants
net.EPi= [net.n+2*NG+2*NL+2*NW+1 : net.n+2*NG+2*NL+2*NW+2*NP]; %PV plants

% the above index sets in Complex region
net.EGi_c = [net.n+1: net.n+NG]; %generators
net.ELi_c = [net.n+NG+1 : net.n+NG+NL]; %loads
net.EWi_c = [net.n+NG+NL+1 : net.n+NG+NL+NW]; %wind plants
net.EPi_c = [net.n+NG+NL+NW+1 : net.n+NG+NL+NW+NP]; %PV plants

net.nDAE = length([net.EGi, net.ELi, net.EWi, net.EPi]);

%dimension of state of whole system
n = net.n + net.nDAE;

% inertia matrix
Mwo = blkdiag(eye(net.n), zeros(net.nDAE));

index = 0;
for j=1:NW
  win{j}.si = [net.ng + index+1: net.ng + index+win{j}.n]; 
  index = index + win{j}.n;  
end
index = 0;
for j=1:NP
  pv{j}.si = [net.ng+net.nw+ index+1: net.ng+net.nw+ index+pv{j}.n]; 
  index = index + pv{j}.n;  
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% output for plot
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
out.del = zeros(1,NG);
out.ome = zeros(1,NG);
out.E = zeros(1,NG);
for j=1:NG
  out.del(j) = gpar{j}.si(1);
  out.ome(j) = gpar{j}.si(2);
  out.E(j) = gpar{j}.si(3);
end
if NW>0
  out.w_wl = zeros(1,NW);
  out.w_wr = zeros(1,NW);
  out.w_thT = zeros(1,NW);
  out.w_is = zeros(1,NW);
  out.w_ir = zeros(1,NW);
  out.w_ig = zeros(1,NW); 
  out.w_vdc = zeros(1,NW); 

  if(sto_flag==1); out.vb = zeros(1,NW); out.idcP = zeros(1,NW); end
  for j=1:NW
    out.w_wl(j) = win{j}.si(1);
    out.w_wr(j) = win{j}.si(2);
    out.w_thT(j) = win{j}.si(3);
    out.w_is(2*j-1:2*j) = win{j}.si(6:7);
    out.w_ir(2*j-1:2*j) = win{j}.si(4:5);
    out.w_ig(2*j-1:2*j) = win{j}.si(8:9);

    if RSC_flag == 1
      out.w_vdc(j) = win{j}.si(18);
    else
      out.w_vdc(j) = win{j}.si(16);
      if sto_flag == 1
	out.vb(j) = win{j}.si(17);
	out.idcP(j) = win{j}.si(18);
      end
    end
  end
end

if NP>0
  out.pv_ig = zeros(1,2*NP);
  out.pv_vdc = zeros(1,NP);
  for j=1:NP
    out.pv_vdc(j) = pv{j}.si(7);
    out.pv_ig(2*j-1:2*j) = pv{j}.si([1:2]);
  end
end