%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Created by Tomonori Sadamoto.
% Updated at 13, Sep. 2018
%
% state variables are defined as follows
% x := [syncgens; wind farms; solar farms; volt of syncgens ; volt. of loads; volt. of wind farms; volt. of solar farms]; 
%
% Note that all complex variables in this simulation are expressed as "augmented vector".
% For example, a complex-valued variable x = a +jb is described as a two-dimensional vector [a; b]. 
% The operation for this augmented variables are defined in the folder "./tools/".
% For example, abs(x) is defined as "./tools/augreal_abs.m"
%
% Run this mfile to reproduce all simulation results in Section "Numerical Simulations"
% Some initial settings that need to be changed accoroding to what you want to see
% are summarized in manual.pdf
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clear all; close all; pause(0.1); addpath('./tool/');
global net NG NL NW NP gpar dae Vref vfdstar win flag mpc pm sims
global compw lin delPW ystar init_PL init_EL init_PP init_IP YL x0_dae Yk n_ode ff Ykc init_EW 
global net_f fcase Yf Yf_wof init_x
global ir_s is_s ws vdc_s Q_s
global Yg Yg_iw Yg_ipv wr_s
global P_spv Q_spv ipv_s mG_s pv flag_pv Spv_s
global compw_pv
global flag_g addPSSu gl sys 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Select a power network to be simulated
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% **** penetration level ****
gamma = [60]; %wind farm
gamma_pv = [200]; %pv farm (200)

% **** load wind & storage ****
data_wind01; 
data_pv01; 

% **** parameters of retrofit controller ****
gains = 100;
gain_i = [1];
nhat = [16];
nhat_pv = [7];

% fault bus
sims.fbus = 10;

% **** load other settings ****
data_68bus; 

if sto_flag == 1
  nhat = 18;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Power flow calculation
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
pflow;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Pre-calculation to define dynamics of individual components
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
precalc;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Calculate equilibrium of each component 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
init_state; 

% error check
flag = -1*ones(1,NW); %-1: use traditional controller
flag_pv = -1*ones(1,NP); %-1: use traditional controller
flag_g = -1; %without use global controller
ff = 0; %0: power line fault flag off
n_ode = n; %dimension of whole system
delPW = [0;0]; %must be 0
x0_dae = init_x; %equilibrium
dx = psm_15s(0, init_x);
if(norm(dx) > 1e-5); error('failed determining initial state'); end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Setup 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
sims.x0type = 5;

%make disturbance of initial state
if(sims.x0type == 2)
  % ***** disturbance on solar farm *****
  ds = zeros(n,1);
  for j=1:NP
    ds(pv{j}.si(1:2)) = -0.7*init_x(pv{j}.si(1:2)); 
  end
  save('ds_pv01.mat','ds'); 

elseif(sims.x0type == 5)
  % ***** disturbance on wind farm *****
  ds = zeros(n,1);
  ds(win{1}.si(1)) = -0.7*init_x(win{1}.si(1)); 

elseif(sims.x0type == -1)
  % ***** 3-phase fault *****
  ds = zeros(n,1); 
  fprintf('[ Fault Simulation (Bus = %d)]\n', sims.fbus);
  
  % do settings
  make_fault_settings; 
  
  % ***** determine initial disturbance *****
  % Note:  state = genstate; EL; EW; EP and ELF
  n_ode = n; 
  flag = -1*ones(1,NW); %traditional controller
  flag_pv = -1*ones(1,NP); %-1: use traditional controller
  flag_g = -1; %without use global controller
  x0_dae = init_x; %equilibrium
  ff = 1; %fault flag on
  options = odeset('Mass',Mf,'RelTol',1e-8,'AbsTol',1e-8);
  [tf,xf] = ode15s(@psm_15s,[-sims.ft:0.001:0], init_xf, options);sims.odemem = 0;
  ds = xf(end, 1:n)'-init_x(1:n); 
  tf = tf(1:end-1); xf = xf(1:end-1, :); 
  ff = 0; 
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Linearization 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
linearization_all;

for j=1:NW
  linearization; 
end
for j=1:NP
  linearization_pv; 
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Simulation without any control
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fprintf('====== Nonlinear simulation ======\n'); 
flag = -1*ones(1,NW); %traditional controller
flag_pv = -1*ones(1,NP); %-1: use traditional controller
flag_g = -1; %without use global controller
x0_dae = init_x; %equilibrium
n_ode = n; 

options = odeset('Mass',Mwo,'RelTol', tol,'AbsTol', tol); 
[to,xo] = ode15s(@psm_15s,[0:0.01:sims.te], init_x+ds, options);sims.odemem = 0;

if(sims.x0type == -1); to = [tf; to]; xo = [xf(:,1:n); xo]; end
% add pre fault trajectories for journal
[to,xo] = add_prefault_traj(to,xo,-1, init_x); 

% calculate some physical quantities
t=to; x=xo;
cal_sg; if(NW>0); cal_wind; end; if(NP>0); cal_pv; end; 

% save
res_ori.t = to;
res_ori.ome = ome;
res_ori.thdiff = thdiff;
res_ori.vG = vabs_G; 
res_ori.vL = vabs_L; 
if NP>0
  res_ori.ig_pv = ig_pv; 
  res_ori.vdc_pv = vdc_pv; 
  res_ori.mG = mGpv;
  res_ori.PP = PP;
end
if NW>0
  res_ori.wl = wl;
  res_ori.wr = wr;
  res_ori.thT = thT;
  res_ori.is = is;
  res_ori.ir = ir;
  res_ori.P = PW;
  res_ori.ig = ig;
  res_ori.vdc = vdc;
  res_ori.mG = mG;
  if sto_flag==1
    res_ori.vb = vb;
  end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Retrofit Controller Design 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
for lp=1:length(gain_i)
  gain = gains(gain_i(lp));
  design_compensators; 
  result{lp}.E = wholew.E;
  result{lp}.vec = wholew.vec;  
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Global Controller Design 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
global_eig; 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Simulation 
% change simulation type depending on your purpose; see manual
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
show_type = 3; 

if show_type ==1
  % ******************
  %  single retrofit on wind (Fig. 18, 19)
  % ******************
  sim_singleRetrofit;   
  pl_art01;

elseif show_type ==2
  % ******************
  %  compare single retrofit vs multiple retrofit (Fig 21)
  % ******************
  sim_singleRetrofit;   
  sim_allRetrofit; 
  pl_art02;

elseif show_type ==3
  % ******************
  %  compare no/all retrofit/global/ all retrfoti+global (Fig 22)
  % ******************
  sim_allRetrofit; 
  sim_Global; 
  sim_allRetrofit_Global; 
  pl_art03;
  
end
